#ifndef __DM_OS_H__
#define __DM_OS_H__

/**
 *
 * \file        os.h
 *
 * \brief       Generic, very simple operating system interface
 *
 * \author      Pete McCormick
 *
 * \date        12/5/2007
 *
 * If this interface does not provide sufficient functionality,
 * we need to extend it since the code should be os independent
 */

////////////////////////////////////////////////////////////////////////////////

#include "dm_nutypedefs.h"

#ifdef __cplusplus
extern "C" {
#endif
////////////////////////////////////////////////////////////////////////////////

#define OS_TASK_EXTRA_BYTES             16

#define OS_ISR_FLAG_NO_MORE_DATA   0x80000000
#define OS_ISR_FLAG_TASK_WOKEN     0x40000000
#define OS_WAIT_FOREVER            0xFFFFFFFF
#define OS_NO_WAIT                 0x0
////////////////////////////////////////////////////////////////////////////////

typedef void (*TASKFUNC)(UINT32 param);

////////////////////////////////////////////////////////////////////////////////

extern volatile int InIsr;

////////////////////////////////////////////////////////////////////////////////

// really simple OS encapsulation

// if periodic, OS will automatically call from delay loop
// maybe have a queue task also? create an associated queue automatically
UINT32 OsCreateNamedPriorityTask(TASKFUNC entryPoint, UINT32 periodMsec, UINT32 queueItemCnt, UINT32 * pParam, UINT8 * pExtra, const signed char * const pName,UINT32 priority,UINT32 stack_size) ;
UINT32 OsCreateNamedTask(TASKFUNC entryPoint, UINT32 periodMsec, UINT32 queueItemCnt, UINT32 * pParam, UINT8 * pExtra, const signed char * const pName) ;
UINT32 OsCreateTask(TASKFUNC entryPoint, UINT32 periodMsec, UINT32 queueItemCnt, UINT32 * pParam, UINT8 * pExtra);
UINT32 OsCreateNamedAdvTask(TASKFUNC entryPoint, UINT16 usStackDepth, UINT32 * pParam, UINT32 uxPriority, const signed char * const pName) ;
UINT32 OsCreateNamedAdvTaskPri(TASKFUNC entryPoint, UINT16 usStackDepth, UINT32 * pParam, UINT32 uxPriority, const signed char * const pName) ; //force priorities on switch platform
UINT32 OsCreateAdvTask(TASKFUNC entryPoint, UINT16 usStackDepth, UINT32 * pParam, UINT32 uxPriority);
void   OsDeleteAdvTask (UINT32 taskHandle);
void   OsSuspendAdvTask(UINT32 taskHandle);
void   OsResumeAdvTask(UINT32 taskHandle);
void OsOneShotTask(UINT32 hFunctionEntryPoint);

void OsDeleteLock(UINT32 lHandle);
void OsCreateLock(UINT32 * pHandle);
INT32 OsLock(UINT32 handle);
INT32 OsLockWait(UINT32 handle, UINT32 waitTicks);
INT32 OsUnlock(UINT32 handle);
UINT32 OsQueueCreate(UINT32 itemCnt, UINT32 lItemSize);
UINT32 OsQueueCreateEx(UINT32 itemCnt, UINT32 lItemSize, char *Name);
UINT32 OsQueueCreateVar(UINT32 itemCnt, UINT32 lItemSize);
INT32 OsQueue(UINT32 handle, void* item, UINT32 timeout);
INT32 OsDequeue(UINT32 handle, void* item);
void OsQueueClear(void);
void  OsQueueDelete(INT32 handle);
void HwDelayMsec(UINT32 ms);
UINT32 HwGetMsec(void);
void OsStart(void);
//void OsTick(void);
UINT16 OsGetDefaultStackVarCnt(void);

// Event imitation
INT32 OsDequeueWait(UINT32 handle, void * item, UINT32 timeout, UINT32 no_consume);
INT32 OsDequeueWait_wStatus (UINT32 handle, void * item, UINT32 timeout, UINT32 no_consume);

//RWC: This is currently only implemented on the STR9, feel free to implement elsewhere
UINT32 OsQueueIsFull( UINT32 handle );
UINT32 OsGetQueueCount( UINT32 handle );

// MNT - 7/28/2008 - Add Counting Semaphore related functions
UINT32 OSCreateCountingSemaphore( unsigned val, unsigned initialCount );
UINT32 OSDeleteCountingSemaphore( UINT32 semId);
UINT32 OSTakeCountingSemaphore(UINT32 semId, UINT32 timOut);
UINT32 OSGiveCountingSemaphore(UINT32 semId);

// simple timer functions
void OsInitTimerTask( UINT16 sExtraStackIncrease );
void *OsStartTimer(INT32 lTimeout, void *pCallback, INT32 bPeriodic);
void *OsStartTimerPri(INT32 lTimeout, void *pCallback, INT32 bPeriodic, UINT32 priority);
void OsCancelTimer(void *hTimer);
void OsUpdateTimer(void *hTimer, INT32 iNewTime);
void OsRetriggerTimer(void *hTimer, INT32 iNewTime);

// Critical Region - USE CAREFULLY
void OsEnterCritical(void);
void OsExitCritical(void);
void* OsAllocMemory(UINT32 byteCnt);
void* OsCallocMemory( UINT32 byteCnt, UINT32 elementSize );
INT32 OsFreeMemory(void *);

UINT32 OsGetCurrentTaskHandle(void);

// pointer to timer callback
typedef void (*pTimer)(void);
#define OS_TIMER_PERIODIC 1
#define OS_TIMER_ONE_TIME 0

// Operating System Errors
enum
{
    ERR_OS_FAIL_TASK_QUEUE,              // 0 unable to create task with queue
    ERR_OS_FAIL_ADV_TASK,                // 1 unable to create advanced task
    ERR_OS_FAIL_LOCK,                    // 2 unable to create Lock
    ERR_OS_FAIL_QUEUE,                   // 3 unable to create queue
    ERR_OS_FAIL_TIMER_TASK,              // 4 unable to create timer task
    ERR_OS_FAIL_TIMER,                   // 5 unable to get memory for timer entry
    ERR_OS_FAIL_STACK,                   // 6 stack corruption or overflow
    ERR_OS_FAIL_INTERRUPT,               // 7 unexpected interrupt
    ERR_OS_FAIL_CNTSEM,                  // 8 could not create counting semaphore
    ERR_OS_FAIL_MEMMGR,                  // 9 could not allocate or deallocate memory from memmgr
    ERR_OS_FAIL_TIMER_TASK_OVERLAP,      // 10 Timer execution longer than 10ms period
    ERR_OS_FAIL_SHELL,                   // 11 shell error
    ERR_OS_FAIL_CONSOLE,                 // 12 console error
    ERR_OS_FAIL_GPIO,                    // 13 GPIO error
    ERR_OS_FAIL_TASK_INFO_PRINT,         // 14 Task info print error
    MAX_DM_ERR_OS
};


// Event Group Related Functions
UINT32 OsCreateEventGroup(void);
UINT32 OsSetEvent(UINT32 eventHandle, UINT32 mask);
UINT32 OsWaitForEventAndClear(UINT32 eventHandle, UINT32 mask);

// run from Idle task ... Cannot do anything that blocks in this task.
void vApplicationIdleHook( void );

// this variable is used to dynamically select what we use for the timer tick
// currently only implemented on STR9 processor
void OsSetTimerSource(UINT8 TickSource);
extern unsigned char OSTickSource;
enum
{
  USE_WDG_FOR_TICK, // use watchdog for OS Tick
  USE_TMR0_FOR_TICK // use timer 0 as OS tick
};

BOOL OsCreateEvent(UINT32* EventHandle);
void OsDeleteEvent(UINT32 EventHandle);
void OsSignalEvent(UINT32 EventHandle);
INT32 OsSendEvent(UINT32 EventHandle);
BOOL OsWaitEvent(UINT32 EventHandle, UINT32 WaitTime);

//WinCE specific methods
INT32 DmSetThreadPriority( UINT16 iPriority );
INT32 DmSetSpecificThreadPriority( UINT32 iThread, UINT16 iPriority );
void DmLowerThreadPriority( void );
void DmIncreaseThreadPriority( void );

UINT16 OsPrintTaskInfo(BOOL console, UINT8* pBuf, UINT16 bufSize);
void OsPrintTaskInfoAdvanced(void);

#ifdef __cplusplus
}
#endif
#endif

